<?php
require_once '../config.php';

if(!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$user = $conn->query("SELECT username FROM users WHERE id = $user_id")->fetch_assoc();

// Get filter parameters
$type_filter = isset($_GET['type']) ? sanitize($_GET['type']) : '';
$status_filter = isset($_GET['status']) ? sanitize($_GET['status']) : '';
$search = isset($_GET['search']) ? sanitize($_GET['search']) : '';
$start_date = isset($_GET['start_date']) ? sanitize($_GET['start_date']) : '';
$end_date = isset($_GET['end_date']) ? sanitize($_GET['end_date']) : '';

// Build query
$where_conditions = ["user_id = $user_id"];
$params = [];
$types = "";

if($type_filter && $type_filter != 'all') {
    $where_conditions[] = "type = ?";
    $params[] = $type_filter;
    $types .= "s";
}

if($status_filter && $status_filter != 'all') {
    $where_conditions[] = "status = ?";
    $params[] = $status_filter;
    $types .= "s";
}

if($search) {
    $where_conditions[] = "(description LIKE ? OR id LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $types .= "ss";
}

if($start_date) {
    $where_conditions[] = "DATE(created_at) >= ?";
    $params[] = $start_date;
    $types .= "s";
}

if($end_date) {
    $where_conditions[] = "DATE(created_at) <= ?";
    $params[] = $end_date;
    $types .= "s";
}

$where_sql = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get transactions
$transactions_sql = "SELECT * FROM transactions 
                     $where_sql 
                     ORDER BY created_at DESC";
$transactions_stmt = $conn->prepare($transactions_sql);
if($types) {
    $transactions_stmt->bind_param($types, ...$params);
}
$transactions_stmt->execute();
$transactions_result = $transactions_stmt->get_result();

// Get statistics
$stats = $conn->query("
    SELECT 
        COUNT(*) as total_count,
        SUM(CASE WHEN type = 'deposit' AND status = 'completed' THEN amount ELSE 0 END) as total_deposits,
        SUM(CASE WHEN type = 'withdrawal' AND status = 'completed' THEN amount ELSE 0 END) as total_withdrawals,
        SUM(CASE WHEN type = 'referral_bonus' AND status = 'completed' THEN amount ELSE 0 END) as total_referral_bonus
    FROM transactions 
    $where_sql
")->fetch_assoc();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transaction Report - Project Sale</title>
    <style>
        @media print {
            body { margin: 0; padding: 20px; }
            .no-print { display: none !important; }
            .break-before { page-break-before: always; }
        }
        
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
        }
        
        .report-header {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 2px solid #333;
            padding-bottom: 20px;
        }
        
        .report-title {
            font-size: 24px;
            font-weight: bold;
            color: #333;
        }
        
        .report-subtitle {
            color: #666;
            margin-top: 5px;
        }
        
        .user-info {
            margin-bottom: 20px;
            padding: 15px;
            background: #f5f5f5;
            border-radius: 5px;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 15px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            padding: 15px;
            border-radius: 5px;
            text-align: center;
            border: 1px solid #ddd;
        }
        
        .stat-value {
            font-size: 24px;
            font-weight: bold;
            margin: 10px 0;
        }
        
        .stat-label {
            font-size: 14px;
            color: #666;
        }
        
        .transactions-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        
        .transactions-table th {
            background: #f5f5f5;
            padding: 12px;
            text-align: left;
            border-bottom: 2px solid #333;
            font-weight: bold;
        }
        
        .transactions-table td {
            padding: 10px;
            border-bottom: 1px solid #ddd;
        }
        
        .positive {
            color: #10b981;
            font-weight: bold;
        }
        
        .negative {
            color: #ef4444;
            font-weight: bold;
        }
        
        .status-completed {
            color: #10b981;
            font-weight: bold;
        }
        
        .status-pending {
            color: #f59e0b;
            font-weight: bold;
        }
        
        .status-failed {
            color: #ef4444;
            font-weight: bold;
        }
        
        .print-footer {
            margin-top: 50px;
            padding-top: 20px;
            border-top: 1px solid #ddd;
            text-align: center;
            color: #666;
            font-size: 14px;
        }
        
        .print-button {
            background: #3b82f6;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            margin: 20px auto;
            display: block;
        }
        
        .print-button:hover {
            background: #2563eb;
        }
    </style>
</head>
<body>
    <!-- Print Button -->
    <div class="no-print" style="text-align: center; margin: 20px;">
        <button onclick="window.print()" class="print-button">
            <i class="fas fa-print"></i> Print Report
        </button>
        <button onclick="window.close()" class="print-button" style="background: #6b7280; margin-left: 10px;">
            <i class="fas fa-times"></i> Close
        </button>
    </div>
    
    <!-- Report Header -->
    <div class="report-header">
        <div class="report-title">Transaction Report</div>
        <div class="report-subtitle">Project Sale - Financial Summary</div>
        <div class="report-subtitle">Generated on: <?php echo date('F d, Y h:i A'); ?></div>
    </div>
    
    <!-- User Information -->
    <div class="user-info">
        <strong>User:</strong> <?php echo $user['username']; ?><br>
        <strong>Report Period:</strong> 
        <?php 
        echo $start_date ? date('M d, Y', strtotime($start_date)) : 'All time';
        echo ' to ';
        echo $end_date ? date('M d, Y', strtotime($end_date)) : 'Present';
        ?><br>
        <strong>Filters Applied:</strong> 
        <?php 
        $filters = [];
        if($type_filter) $filters[] = "Type: " . ucfirst($type_filter);
        if($status_filter) $filters[] = "Status: " . ucfirst($status_filter);
        if($search) $filters[] = "Search: " . $search;
        echo $filters ? implode(', ', $filters) : 'None';
        ?>
    </div>
    
    <!-- Statistics -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-label">Total Transactions</div>
            <div class="stat-value"><?php echo $stats['total_count']; ?></div>
        </div>
        
        <div class="stat-card">
            <div class="stat-label">Total Deposits</div>
            <div class="stat-value positive">৳<?php echo number_format($stats['total_deposits'], 2); ?></div>
        </div>
        
        <div class="stat-card">
            <div class="stat-label">Total Withdrawals</div>
            <div class="stat-value negative">৳<?php echo number_format($stats['total_withdrawals'], 2); ?></div>
        </div>
        
        <div class="stat-card">
            <div class="stat-label">Referral Earnings</div>
            <div class="stat-value positive">৳<?php echo number_format($stats['total_referral_bonus'], 2); ?></div>
        </div>
    </div>
    
    <!-- Transactions Table -->
    <h3>Transaction Details</h3>
    <table class="transactions-table">
        <thead>
            <tr>
                <th>ID</th>
                <th>Type</th>
                <th>Amount</th>
                <th>Description</th>
                <th>Status</th>
                <th>Date</th>
            </tr>
        </thead>
        <tbody>
            <?php 
            $row_count = 0;
            while($transaction = $transactions_result->fetch_assoc()): 
                $row_count++;
                if($row_count % 25 == 0) {
                    echo '<tr class="break-before"></tr>';
                }
                
                $is_positive = in_array($transaction['type'], ['deposit', 'referral_bonus']);
                $amount_class = $is_positive ? 'positive' : 'negative';
                $amount_sign = $is_positive ? '+' : '-';
                
                $status_class = 'status-' . $transaction['status'];
            ?>
            <tr>
                <td>#<?php echo str_pad($transaction['id'], 6, '0', STR_PAD_LEFT); ?></td>
                <td><?php echo ucfirst(str_replace('_', ' ', $transaction['type'])); ?></td>
                <td class="<?php echo $amount_class; ?>">
                    <?php echo $amount_sign; ?>৳<?php echo number_format($transaction['amount'], 2); ?>
                </td>
                <td><?php echo htmlspecialchars($transaction['description']); ?></td>
                <td class="<?php echo $status_class; ?>">
                    <?php echo ucfirst($transaction['status']); ?>
                </td>
                <td><?php echo date('M d, Y h:i A', strtotime($transaction['created_at'])); ?></td>
            </tr>
            <?php endwhile; ?>
            
            <?php if($row_count == 0): ?>
            <tr>
                <td colspan="6" style="text-align: center; padding: 20px;">
                    No transactions found
                </td>
            </tr>
            <?php endif; ?>
        </tbody>
    </table>
    
    <!-- Report Summary -->
    <div class="print-footer">
        <div style="margin-bottom: 10px;">
            <strong>Report Summary:</strong><br>
            Total Transactions: <?php echo $stats['total_count']; ?><br>
            Net Balance: ৳<?php 
                $net = $stats['total_deposits'] + $stats['total_referral_bonus'] - $stats['total_withdrawals'];
                echo number_format($net, 2);
            ?>
        </div>
        <div>
            This report was generated by Project Sale System.<br>
            Report ID: <?php echo uniqid(); ?>
        </div>
    </div>
    
    <script>
    // Auto print when page loads
    window.onload = function() {
        setTimeout(function() {
            window.print();
        }, 1000);
    };
    </script>
</body>
</html>